<?php
require '../config/sessionToken.php'; // Assuming this has the logic to verify session tokens
require_once '../config/constants.php';

// Function to fetch chapters based on class_name and subject
function getChapters($data, $pdo) {
    $sessionToken = $_COOKIE['session_token'] ??'';

    // Verify the session token
    $isValid = Token::Verify($sessionToken, APP_KEY);
    $isValidResponse = json_decode($isValid, true);
    if ($isValidResponse['headers']['status'] == 'failed') {
        return json_encode($isValidResponse);
    }

    // Check if class_name and subject are provided
    if (empty($data['class_name']) || empty($data['subject'])) {
        return json_encode([
            'headers' => [
                'status' => FAILED,
                'responseCode' => MISSING_REQUIRED_FIELDS,
                'message' => MISSING_REQUIRED_FIELDS_TEXT
            ]
        ]);
    }

    $className = $data['class_name'];
    $subject = $data['subject'];

    // Check if the class exists
    $classStmt = $pdo->prepare("SELECT id, class_name_bn FROM class WHERE class_name = :class_name");
    $classStmt->bindParam(':class_name', $className);
    $classStmt->execute();
    $classExists = $classStmt->fetch(PDO::FETCH_ASSOC);

    if (!$classExists) {
        return json_encode([
            'headers' => [
                'status' => FAILED,
                'responseCode' => CLASS_DOES_NOT_EXIST_CODE,
                'message' => CLASS_DOES_NOT_EXIST
            ]
        ]);
    }

    // Check if the subject exists for the provided class
    $subjectStmt = $pdo->prepare("SELECT id, subject_name_bn FROM subjects WHERE class_name = :class_name AND subject_name = :subject_name");
    $subjectStmt->bindParam(':class_name', $className);
    $subjectStmt->bindParam(':subject_name', $subject);
    $subjectStmt->execute();
    $subjectExists = $subjectStmt->fetch(PDO::FETCH_ASSOC);

    if (!$subjectExists) {
        return json_encode([
            'headers' => [
                'status' => FAILED,
                'responseCode' => SUBJECT_DOES_NOT_EXIST_FOR_CLASS_CODE,
                'message' => SUBJECT_DOES_NOT_EXIST_FOR_CLASS
            ]
        ]);
    }

    // Fetch chapters for the specified class and subject
    $chapterStmt = $pdo->prepare("SELECT * FROM chapters WHERE class_name = :class_name AND subject = :subject");
    $chapterStmt->bindParam(':class_name', $className);
    $chapterStmt->bindParam(':subject', $subject);
    $chapterStmt->execute();
    $chapters = $chapterStmt->fetchAll(PDO::FETCH_ASSOC);

    if ($chapters) {
        // Loop through the chapters and add class_name_bn and subject_name_bn
        foreach ($chapters as &$chapter) {
            $chapter['class_name_bn'] = $classExists['class_name_bn']; // Add the Bengali class name
            $chapter['subject_name_bn'] = $subjectExists['subject_name_bn']; // Add the Bengali subject name
        }

        return json_encode([
            'headers' => [
                'status' => SUCCESS,
                'responseCode' => COMMON_SUCCESS,
                'message' => CHAPTERS_RETERIVED_SUCCESS
            ],
            'body' => [
                'chaptersList' => $chapters
            ]
        ]);
    } else {
        return json_encode([
            'headers' => [
                'status' => FAILED,
                'responseCode' => NO_CHAPTERS_FOUND_FOR_CLASS_SUBJECT_CODE,
                'message' => NO_CHAPTERS_FOUND_FOR_CLASS_SUBJECT
            ],
            'body' => [
                'chaptersList' => []
            ]
        ]);
    }
}


// Get input data from the POST request
$input = json_decode(file_get_contents('php://input'), true);

$locale = $input['locale'] ?? 'en';


    if ($locale === 'bn') {
        require_once '../config/app_bn.php';
    } else {
        require_once '../config/app_en.php';
    }   


// If no data is provided, return an error
if (empty($input)) {
    echo json_encode([
        'headers' => [
            'status' => FAILED,
            'responseCode' => MISSING_REQUIRED_FIELDS,
            'message' => MISSING_REQUIRED_FIELDS_TEXT
        ]
    ]);
    exit();
}

// Call the function to get the chapters
echo getChapters($input, $pdo);
?>
