<?php
require '../config/sessionToken.php'; // Assuming this has the logic to verify session tokens
require_once '../config/constants.php';

// Function to fetch all subjects by class_name and add question count and class_name_bn
function getSubjectsByClass($data, $pdo) {
    $sessionToken = $_COOKIE['session_token'] ?? '';

    // Verify the session token
    $isValid = Token::Verify($sessionToken, APP_KEY);
    $isValidResponse = json_decode($isValid, true);
    if ($isValidResponse['headers']['status'] == 'failed') {
        return json_encode($isValidResponse);
    }

    // Check if class_name is set
    if (empty($data['class_name'])) {
        return json_encode([
            'headers' => [
                'status' => FAILED,
                'responseCode' => MISSING_REQUIRED_FIELDS,
                'message' => MISSING_REQUIRED_FIELDS_TEXT
            ]
        ]);
    }

    // Fetch the class with class_name_bn
    $stmt = $pdo->prepare("SELECT id, class_name_bn FROM class WHERE class_name = :class_name");
    $stmt->bindParam(':class_name', $data['class_name']);
    $stmt->execute();
    $existingClass = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$existingClass) {
        return json_encode([
            'headers' => [
                'status' => FAILED,
                'responseCode' => CLASS_DOES_NOT_EXIST_CODE,
                'message' => CLASS_DOES_NOT_EXIST
            ]
        ]);
    }

    // Fetch all subjects for the given class
    $stmt = $pdo->prepare("SELECT * FROM subjects WHERE class_name = :class_name");
    $stmt->bindParam(':class_name', $data['class_name']);
    $stmt->execute();
    $subjects = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Check if any subjects exist
    if ($subjects) {
        // Add question count for each subject and include class_name_bn
        foreach ($subjects as &$subject) {
            $stmt = $pdo->prepare("SELECT COUNT(*) AS noOfQuestions FROM questions WHERE subject_name = :subject_name");
            $stmt->execute(['subject_name' => $subject['subject_name']]);
            $questionCount = $stmt->fetch(PDO::FETCH_ASSOC);
            $subject['noOfQuestions'] = $questionCount['noOfQuestions'] ?? 0;
            $subject['class_name_bn'] = $existingClass['class_name_bn']; // Include class_name_bn for each subject
        }

        return json_encode([
            'headers' => [
                'status' => SUCCESS,
                'responseCode' => SUBJECTS_RETERIVED_SUCCESS_CODE,
                'message' => SUBJECTS_RETERIVED_SUCCESS
            ],
            'body' => [
                'subjectsList' => $subjects
            ]
        ]);
    } else {
        return json_encode([
            'headers' => [
                'status' => SUCCESS,
                'responseCode' => SUBJECTS_RETERIVED_FAILED_CODE,
                'message' => 'No subjects found for this class'
            ],
            'body' => []
        ]);
    }
}

// Function to get the Authorization token from headers
function getAuthorizationToken() {
    $headers = getallheaders();
    return isset($headers['Authorization']) ? trim(str_replace('Bearer ', '', $headers['Authorization'])) : null;
}

// Get input data from the POST request
$input = json_decode(file_get_contents('php://input'), true);

$locale = $input['locale'] ?? 'en';

if ($locale === 'bn') {
    require_once '../config/app_bn.php';
} else {
    require_once '../config/app_en.php';
}

// If no data is provided, return an error
if (empty($input)) {
    echo json_encode([
        'headers' => [
            'status' => FAILED,
            'responseCode' => MISSING_REQUIRED_FIELDS,
            'message' => MISSING_REQUIRED_FIELDS_TEXT
        ]
    ]);
    exit();
}

// Call the function to get subjects by class
echo getSubjectsByClass($input, $pdo);
?>
